import bpy

# Add-on Preferences (colors)
class ShaderSwitcherAddonPreferences(bpy.types.AddonPreferences):
    # IMPORTANT: use the top-level package name for bl_idname
    bl_idname = __package__

    color_switch: bpy.props.FloatVectorProperty(
        name="Shader Switcher", subtype='COLOR', size=3, min=0.0, max=1.0, default=(0.086, 0.157, 0.247),
        description="Node color for Shader Switcher nodes")
    color_mix: bpy.props.FloatVectorProperty(
        name="Shader Mixer", subtype='COLOR', size=3, min=0.0, max=1.0, default=(0.086, 0.247, 0.157),
        description="Node color for Shader Mixer nodes")
    color_random: bpy.props.FloatVectorProperty(
        name="Shader Randomizer", subtype='COLOR', size=3, min=0.0, max=1.0, default=(0.247, 0.157, 0.086),
        description="Node color for Shader Randomizer nodes")
    color_colormixer: bpy.props.FloatVectorProperty(
        name="Color Mixer", subtype='COLOR', size=3, min=0.0, max=1.0, default=(0.247, 0.086, 0.157),
        description="Node color for Color Mixer nodes")
    color_master: bpy.props.FloatVectorProperty(
        name="Master Material", subtype='COLOR', size=3, min=0.0, max=1.0, default=(0.157, 0.086, 0.247),
        description="Node color for Master Material nodes")

    def draw(self, context):
        layout = self.layout
        box = layout.box()
        box.label(text="Node Colors", icon='COLOR')
        col = box.column(align=True)
        col.prop(self, "color_switch")
        col.prop(self, "color_mix")
        col.prop(self, "color_random")
        col.prop(self, "color_colormixer")
        col.prop(self, "color_master")

# Sidebar Panel
class NODE_PT_shader_switcher_panel(bpy.types.Panel):
    bl_space_type = 'NODE_EDITOR'
    bl_region_type = 'UI'
    bl_category = 'Shader Switcher'
    bl_label = "Shader/Color Switcher Tools"

    @classmethod
    def poll(cls, context):
        return (context.space_data and
                context.space_data.type == 'NODE_EDITOR' and
                context.space_data.tree_type == 'ShaderNodeTree')

    def draw(self, context):
        layout = self.layout

        # Create
        create_box = layout.box()
        create_box.label(text="Create New Node", icon='NODE')
        addon_props = context.scene.shader_switcher_addon_props

        row = create_box.row(align=True)
        row.prop(addon_props, "creation_type", text="")
        row.prop(addon_props, "initial_input_count", text="Inputs")

        op = create_box.operator("node.add_shader_switcher", icon='ADD', text="Add Node")
        op.node_type = addon_props.creation_type
        op.initial_count = addon_props.initial_input_count

        layout.separator()

        # Manage selected
        manage_box = layout.box()
        manage_box.label(text="Selected Node Controls", icon='TOOL_SETTINGS')

        active_node = context.space_data.edit_tree.nodes.active if context.space_data.edit_tree else None
        is_our = False; target_group = None
        if active_node and active_node.type == 'GROUP' and active_node.node_tree:
            if "mode" in active_node.node_tree:
                is_our = True; target_group = active_node.node_tree

        if is_our:
            manage_box.label(text=f"Active: {active_node.label}", icon='DOT')
            row = manage_box.row(); row.prop(target_group.shader_switcher_props, "mode", text="Node Type")

            if target_group.users > 1:
                box = manage_box.box(); box.alert = True
                box.label(text="This node group is linked!", icon='INFO')
                box.label(text="Editing will affect ALL users. Adding/Removing inputs makes it single-user.", icon='ERROR')

            manage_box.operator("node.add_shader_switcher_input", icon="ADD", text="Add Input")

            mode = target_group.get("mode")
            if mode:
                input_management_box = manage_box.box()
                input_management_box.label(text="Manage Individual Inputs:", icon='REMOVE')

                primary_inputs = []
                if mode in ['Switch', 'Mix', 'Random', 'Master']:
                    primary_inputs = [s for s in active_node.inputs if s.name.startswith('Input ')]
                elif mode == 'ColorMixer':
                    primary_inputs = [s for s in active_node.inputs if s.name.startswith('Color ')]

                primary_inputs.sort(key=lambda s: int(s.name.split(' ')[1]))

                if not primary_inputs:
                    input_management_box.label(text="No inputs to manage.", icon='INFO')
                else:
                    for socket in primary_inputs:
                        row = input_management_box.row(align=True)
                        row.label(text=socket.name)
                        group_current_count = target_group.get("input_count", 0)
                        min_inputs_for_mode = 2 if mode == 'Random' else 1
                        can_remove = group_current_count > min_inputs_for_mode
                        op_remove = row.operator("node.remove_shader_input", text="", icon='TRASH')
                        op_remove.input_index_to_remove = int(socket.name.split(' ')[1])
                        row.enabled = can_remove
                        if not can_remove:
                            row.label(text="Min inputs reached")

            # Node-specific UI
            if mode and mode != 'Master':
                control_box = manage_box.box()
                control_box.label(text="Node Specific Inputs:", icon='PROPERTIES')

                if mode == 'Switch':
                    if "Index" in active_node.inputs:
                        control_box.prop(active_node.inputs["Index"], "default_value", text="Active Index")

                elif mode == 'Random':
                    if "Seed" in active_node.inputs:
                        control_box.prop(active_node.inputs["Seed"], "default_value", text="Pattern Seed")

                elif mode == 'Mix':
                    factor_inputs = [s for s in active_node.inputs if s.name.startswith('Factor ')]
                    factor_inputs.sort(key=lambda x: int(x.name.split(' ')[1]), reverse=True)
                    if factor_inputs:
                        sub_col = control_box.column(align=True)
                        for socket in factor_inputs:
                            layer_num = socket.name.split(' ')[1]
                            sub_col.prop(socket, "default_value", text=f"Layer {layer_num} Factor")
                    else:
                        control_box.label(text="No Factor inputs found.")

                elif mode == 'ColorMixer':
                    current_color_count = target_group.get("input_count", 0)
                    for i in reversed(range(current_color_count)):
                        layer_num = i + 1
                        f_name = f"Factor {layer_num}"; c_name = f"Color {layer_num}"
                        if f_name in active_node.inputs and c_name in active_node.inputs:
                            f_socket = active_node.inputs[f_name]
                            c_socket = active_node.inputs[c_name]
                            row = control_box.row(align=True)
                            row.label(text=f"Layer {layer_num}:")
                            main_split = row.split(factor=0.25, align=True)
                            main_split.prop(f_socket, "default_value", text="Fac")
                            color_blend_split = main_split.split(factor=0.6, align=True)
                            color_blend_split.prop(c_socket, "default_value", text="")
                            mix_node_name = target_group.get(f"MixNode_{layer_num}")
                            if mix_node_name and mix_node_name in target_group.nodes:
                                mix_node = target_group.nodes[mix_node_name]
                                if hasattr(mix_node, 'blend_type'):
                                    color_blend_split.prop(mix_node, 'blend_type', text="")
                                else:
                                    color_blend_split.label(text="(Blend Type Missing)")
                            else:
                                color_blend_split.label(text="(Mix node missing)")

            if mode == 'Master':
                assign_box = manage_box.box()
                assign_box.label(text="Assign Selection to Branch", icon='RESTRICT_SELECT_OFF')
                current_count = target_group.get("input_count", 0)
                ids = [int(target_group.get(f"id_{i}", i)) for i in range(1, current_count + 1)]
                if ids:
                    row = None
                    for idx, pid in enumerate(ids):
                        if idx % 6 == 0:
                            row = assign_box.row(align=True)
                        op_btn = row.operator("object.assign_selection_to_branch", text=str(pid))
                        op_btn.pass_index = int(pid)
                else:
                    assign_box.label(text="No branches found.", icon='INFO')
        else:
            manage_box.label(text="No Shader Switcher node selected.", icon='INFO')
            manage_box.label(text="Select a node in the editor to manage its properties.", icon='NODE_SEL')

        layout.separator()
        global_box = layout.box()
        global_box.label(text="Global Tools", icon='FILE_BLEND')
        global_box.operator("node.remove_unused_shader_switchers", icon='TRASH', text="Clean Up Unused Groups")

# Add menu hook
def shader_switcher_menu(self, context):
    self.layout.operator("node.add_shader_switcher", icon="NODE")

# Registration
classes = (
    ShaderSwitcherAddonPreferences,
    NODE_PT_shader_switcher_panel,
)

def register():
    for cls in classes:
        bpy.utils.register_class(cls)

def unregister():
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)
